#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import absolute_import
from __future__ import print_function
from __future__ import division

import tensorflow as tf

from zhusuan import distributions
from zhusuan.framework.bn import StochasticTensor
from zhusuan.legacy.distributions import special


__all__ = [
    'Normal',
    'FoldNormal',
    'Bernoulli',
    'Categorical',
    'OnehotCategorical',
    'Discrete',
    'OnehotDiscrete',
    'Uniform',
    'Gamma',
    'Beta',
    'Poisson',
    'Binomial',
    'InverseGamma',
    'Laplace',
    'MultivariateNormalCholesky',
    'MatrixVariateNormalCholesky',
    'Multinomial',
    'UnnormalizedMultinomial',
    'BagofCategoricals',
    'Dirichlet',
    'BinConcrete',
    'BinGumbelSoftmax',
    'ExpConcrete',
    'ExpGumbelSoftmax',
    'Concrete',
    'GumbelSoftmax',
    'Empirical',
    'Implicit',
]


class Normal(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Normal `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param _sentinel: Used to prevent positional parameters. Internal,
        do not use.
    :param mean: A `float` Tensor. The mean of the Normal distribution.
        Should be broadcastable to match `logstd`.
    :param std: A `float` Tensor. The standard deviation of the Normal
        distribution. Should be positive and broadcastable to match `mean`.
    :param logstd: A `float` Tensor. The log standard deviation of the Normal
        distribution. Should be broadcastable to match `mean`.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        `StochasticTensor` are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 mean=0.,
                 _sentinel=None,
                 std=None,
                 logstd=None,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Normal(
            mean,
            _sentinel=_sentinel,
            std=std,
            logstd=logstd,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Normal, self).__init__(None, name, dist, n_samples=n_samples,
                                     **kwargs)


class FoldNormal(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate FoldNormal `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param mean: A `float` Tensor. The mean of the FoldNormal distribution.
        Should be broadcastable to match `std` or `logstd`.
    :param _sentinel: Used to prevent positional parameters. Internal,
        do not use.
    :param std: A `float` Tensor. The standard deviation of the FoldNormal
        distribution. Should be positive and broadcastable to match `mean`.
    :param logstd: A `float` Tensor. The log standard deviation of the
        FoldNormal distribution. Should be broadcastable to match `mean`.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        `StochasticTensor` are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 mean=0.,
                 _sentinel=None,
                 std=None,
                 logstd=None,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.FoldNormal(
            mean,
            _sentinel=_sentinel,
            std=std,
            logstd=logstd,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(FoldNormal, self).__init__(None, name, dist, n_samples=n_samples,
                                         **kwargs)


class Bernoulli(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Bernoulli `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param logits: A `float` Tensor. The log-odds of probabilities of being 1.

        .. math:: \\mathrm{logits} = \\log \\frac{p}{1 - p}

    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param dtype: The value type of this `StochasticTensor`. Can be
        int (`tf.int16`, `tf.int32`, `tf.int64`) or float (`tf.float16`,
        `tf.float32`, `tf.float64`). Default is `int32`.
    """

    def __init__(self,
                 name,
                 logits,
                 n_samples=None,
                 group_ndims=0,
                 dtype=tf.int32,
                 **kwargs):
        dist = distributions.Bernoulli(
            logits,
            group_ndims=group_ndims,
            dtype=dtype,
            **kwargs
        )
        super(Bernoulli, self).__init__(None, name, dist, n_samples=n_samples,
                                        **kwargs)


class Categorical(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Categorical `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param logits: A N-D (N >= 1) `float` Tensor of shape (...,
        n_categories). Each slice `[i, j,..., k, :]` represents the
        un-normalized log probabilities for all categories.

        .. math:: \\mathrm{logits} \\propto \\log p

    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param dtype: The value type of this `StochasticTensor`. Can be
        `float32`, `float64`, `int32`, or `int64`. Default is `int32`.

    A single sample is a (N-1)-D Tensor with `tf.int32` values in range
    [0, n_categories).
    """

    def __init__(self,
                 name,
                 logits,
                 n_samples=None,
                 group_ndims=0,
                 dtype=tf.int32,
                 **kwargs):
        dist = distributions.Categorical(
            logits,
            group_ndims=group_ndims,
            dtype=dtype,
            **kwargs
        )
        super(Categorical, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)


Discrete = Categorical


class Uniform(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Uniform `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param minval: A `float` Tensor. The lower bound on the range of the
        uniform distribution. Should be broadcastable to match `maxval`.
    :param maxval: A `float` Tensor. The upper bound on the range of the
        uniform distribution. Should be element-wise bigger than `minval`.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        `StochasticTensor` are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 minval=0.,
                 maxval=1.,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Uniform(
            minval,
            maxval,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Uniform, self).__init__(None, name, dist, n_samples=n_samples,
                                      **kwargs)


class Gamma(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Gamma `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param alpha: A `float` Tensor. The shape parameter of the Gamma
        distribution. Should be positive and broadcastable to match `beta`.
    :param beta: A `float` Tensor. The inverse scale parameter of the Gamma
        distribution. Should be positive and broadcastable to match `alpha`.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 alpha,
                 beta,
                 n_samples=None,
                 group_ndims=0,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Gamma(
            alpha,
            beta,
            group_ndims=group_ndims,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Gamma, self).__init__(None, name, dist, n_samples=n_samples,
                                    **kwargs)


class Beta(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Beta `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param alpha: A `float` Tensor. One of the two shape parameters of the
        Beta distribution. Should be positive and broadcastable to match
        `beta`.
    :param beta: A `float` Tensor. One of the two shape parameters of the
        Beta distribution. Should be positive and broadcastable to match
        `alpha`.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 alpha,
                 beta,
                 n_samples=None,
                 group_ndims=0,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Beta(
            alpha,
            beta,
            group_ndims=group_ndims,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Beta, self).__init__(None, name, dist, n_samples=n_samples,
                                   **kwargs)


class Poisson(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Poisson `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param rate: A `float` Tensor. The rate parameter of Poisson
        distribution. Must be positive.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param dtype: The value type of this `StochasticTensor`. Can be
        int (`tf.int16`, `tf.int32`, `tf.int64`) or float (`tf.float16`,
        `tf.float32`, `tf.float64`). Default is `int32`.
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 rate,
                 n_samples=None,
                 group_ndims=0,
                 dtype=tf.int32,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Poisson(
            rate,
            group_ndims=group_ndims,
            dtype=dtype,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Poisson, self).__init__(None, name, dist, n_samples=n_samples,
                                      **kwargs)


class Binomial(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Binomial `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param logits: A `float` Tensor. The log-odds of probabilities.

        .. math:: \\mathrm{logits} = \\log \\frac{p}{1 - p}

    :param n_experiments: A 0-D `int32` Tensor. The number of experiments
        for each sample.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param dtype: The value type of this `StochasticTensor`. Can be
        int (`tf.int16`, `tf.int32`, `tf.int64`) or float (`tf.float16`,
        `tf.float32`, `tf.float64`). Default is `int32`.
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 logits,
                 n_experiments,
                 n_samples=None,
                 group_ndims=0,
                 dtype=tf.int32,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Binomial(
            logits,
            n_experiments,
            group_ndims=group_ndims,
            dtype=dtype,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Binomial, self).__init__(None, name, dist, n_samples=n_samples,
                                       **kwargs)


class MultivariateNormalCholesky(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of multivariate normal `StochasticTensor`, where covariance is
    parameterized with the lower triangular matrix :math:`L` in Cholesky
    decomposition :math:`LL^T = \Sigma`.

    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param mean: An N-D `float` Tensor of shape [..., n_dim]. Each slice
        [i, j, ..., k, :] represents the mean of a single multivariate normal
        distribution.
    :param cov_tril: An (N+1)-D `float` Tensor of shape [..., n_dim, n_dim].
        Each slice [i, ..., k, :, :] represents the lower triangular matrix in
        the Cholesky decomposition of the covariance of a single distribution.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        distribution are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 mean,
                 cov_tril,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.MultivariateNormalCholesky(
            mean,
            cov_tril,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(MultivariateNormalCholesky, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)


class MatrixVariateNormalCholesky(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of matrix variate normal `StochasticTensor`, where covariances
    :math:`U` and :math:`V` are parameterized with the lower triangular
    matrix in Cholesky decomposition,

    .. math ::
        L_u \\text{s.t.} L_u L_u^T = U,\\; L_v \\text{s.t.} L_v L_v^T = V

    See :class:`~zhusuan.model.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param mean: An N-D `float` Tensor of shape [..., n_row, n_col]. Each slice
        [i, j, ..., k, :, :] represents the mean of a single matrix variate
        normal distribution.
    :param u_tril: An N-D `float` Tensor of shape [..., n_row, n_row].
        Each slice [i, j, ..., k, :, :] represents the lower triangular matrix
        in the Cholesky decomposition of the among-row covariance of a single
        matrix variate normal distribution.
    :param v_tril: An N-D `float` Tensor of shape [..., n_col, n_col].
        Each slice [i, j, ..., k, :, :] represents the lower triangular matrix
        in the Cholesky decomposition of the among-column covariance of a
        single matrix variate normal distribution.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        distribution are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 mean,
                 u_tril,
                 v_tril,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.MatrixVariateNormalCholesky(
            mean,
            u_tril,
            v_tril,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(MatrixVariateNormalCholesky, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)


class Multinomial(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of Multinomial `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param logits: A N-D (N >= 1) `float` Tensor of shape `[..., n_categories]`.
        Each slice `[i, j, ..., k, :]` represents the log probabilities for
        all categories. By default (when `normalize_logits=True`), the
        probabilities could be un-normalized.

        .. math:: \\mathrm{logits} \\propto \\log p

    :param n_experiments: A 0-D `int32` Tensor or `None`. When it is a 0-D
        `int32` integer, it represents the number of experiments for each
        sample, which should be invariant among samples. In this situation
        `_sample` function is supported. When it is `None`, `_sample` function
        is not supported, and when calculating probabilities the number of
        experiments will be inferred from `given`, so it could vary among
        samples.
    :param normalize_logits: A bool indicating whether `logits` should be
        normalized when computing probability. If you believe `logits` is
        already normalized, set it to `False` to speed up. Default is True.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param dtype: The value type of this `StochasticTensor`. Can be
        int (`tf.int16`, `tf.int32`, `tf.int64`) or float (`tf.float16`,
        `tf.float32`, `tf.float64`). Default is `int32`.

    A single sample is a N-D Tensor with the same shape as logits. Each slice
    `[i, j, ..., k, :]` is a vector of counts for all categories.
    """

    def __init__(self,
                 name,
                 logits,
                 n_experiments,
                 normalize_logits=True,
                 n_samples=None,
                 group_ndims=0,
                 dtype=tf.int32,
                 **kwargs):
        dist = distributions.Multinomial(
            logits,
            n_experiments,
            normalize_logits=normalize_logits,
            group_ndims=group_ndims,
            dtype=dtype,
            **kwargs
        )
        super(Multinomial, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)


class UnnormalizedMultinomial(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of UnnormalizedMultinomial `StochasticTensor`.
    UnnormalizedMultinomial distribution calculates probabilities differently
    from :class:`Multinomial`: It considers the bag-of-words `given` as a
    statistics of an ordered result sequence, and calculates the probability
    of the (imagined) ordered sequence. Hence it does not multiply the term

    .. math::

        \\binom{n}{k_1, k_2, \\dots} =  \\frac{n!}{\\prod_{i} k_i!}

    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param logits: A N-D (N >= 1) `float` Tensor of shape `[..., n_categories]`.
        Each slice `[i, j, ..., k, :]` represents the log probabilities for
        all categories. By default (when `normalize_logits=True`), the
        probabilities could be un-normalized.

        .. math:: \\mathrm{logits} \\propto \\log p

    :param normalize_logits: A bool indicating whether `logits` should be
        normalized when computing probability. If you believe `logits` is
        already normalized, set it to `False` to speed up. Default is True.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param dtype: The value type of this `StochasticTensor`. Can be
        int (`tf.int16`, `tf.int32`, `tf.int64`) or float (`tf.float16`,
        `tf.float32`, `tf.float64`). Default is `int32`.

    A single sample is a N-D Tensor with the same shape as logits. Each slice
    `[i, j, ..., k, :]` is a vector of counts for all categories.
    """

    def __init__(self,
                 name,
                 logits,
                 normalize_logits=True,
                 group_ndims=0,
                 dtype=tf.int32,
                 **kwargs):
        dist = distributions.UnnormalizedMultinomial(
            logits,
            normalize_logits=normalize_logits,
            group_ndims=group_ndims,
            dtype=dtype,
            **kwargs
        )
        super(UnnormalizedMultinomial, self).__init__(
            None, name, dist, **kwargs)


BagofCategoricals = UnnormalizedMultinomial


class OnehotCategorical(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of one-hot Categorical `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param logits: A N-D (N >= 1) `float` Tensor of shape (...,
        n_categories). Each slice `[i, j, ..., k, :]` represents the
        un-normalized log probabilities for all categories.

        .. math:: \\mathrm{logits} \\propto \\log p

    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param dtype: The value type of this `StochasticTensor`. Can be
        int (`tf.int16`, `tf.int32`, `tf.int64`) or float (`tf.float16`,
        `tf.float32`, `tf.float64`). Default is `int32`.

    A single sample is a N-D Tensor with the same shape as logits. Each slice
    `[i, j, ..., k, :]` is a one-hot vector of the selected category.
    """

    def __init__(self,
                 name,
                 logits,
                 n_samples=None,
                 group_ndims=0,
                 dtype=tf.int32,
                 **kwargs):
        dist = distributions.OnehotCategorical(
            logits,
            group_ndims=group_ndims,
            dtype=dtype,
            **kwargs
        )
        super(OnehotCategorical, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)


OnehotDiscrete = OnehotCategorical


class Dirichlet(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of Dirichlet `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param alpha: A N-D (N >= 1) `float` Tensor of shape (..., n_categories).
        Each slice `[i, j, ..., k, :]` represents the concentration parameter
        of a Dirichlet distribution. Should be positive.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param check_numerics: Bool. Whether to check numeric issues.

    A single sample is a N-D Tensor with the same shape as alpha. Each slice
    `[i, j, ..., k, :]` of the sample is a vector of probabilities of a
    Categorical distribution `[x_1, x_2, ... ]`, which lies on the simplex

    .. math:: \\sum_{i} x_i = 1, 0 < x_i < 1

    """

    def __init__(self,
                 name,
                 alpha,
                 n_samples=None,
                 group_ndims=0,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Dirichlet(
            alpha,
            group_ndims=group_ndims,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Dirichlet, self).__init__(None, name, dist, n_samples=n_samples,
                                        **kwargs)


class InverseGamma(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate InverseGamma `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param alpha: A `float` Tensor. The shape parameter of the InverseGamma
        distribution. Should be positive and broadcastable to match `beta`.
    :param beta: A `float` Tensor. The scale parameter of the InverseGamma
        distribution. Should be positive and broadcastable to match `alpha`.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 alpha,
                 beta,
                 n_samples=None,
                 group_ndims=0,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.InverseGamma(
            alpha,
            beta,
            group_ndims=group_ndims,
            check_numerics=check_numerics,
            **kwargs
        )
        super(InverseGamma, self).__init__(None, name, dist,
                                           n_samples=n_samples, **kwargs)


class Laplace(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate Laplace `StochasticTensor`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param loc: A `float` Tensor. The location parameter of the Laplace
        distribution. Should be broadcastable to match `scale`.
    :param scale: A `float` Tensor. The scale parameter of the Laplace
        distribution. Should be positive and broadcastable to match `loc`.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        `StochasticTensor` are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 loc,
                 scale,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Laplace(
            loc,
            scale,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Laplace, self).__init__(None, name, dist, n_samples=n_samples,
                                      **kwargs)


class BinConcrete(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of univariate BinConcrete `StochasticTensor` from
    (Maddison, 2016). It is the binary case of :class:`~Concrete`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    .. seealso::

        :class:`~Concrete` and :class:`~ExpConcrete`

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param temperature: A 0-D `float` Tensor. The temperature of the relaxed
        distribution. The temperature should be positive.
    :param logits: A `float` Tensor. The log-odds of probabilities of being 1.

        .. math:: \\mathrm{logits} = \\log \\frac{p}{1 - p}

    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        `StochasticTensor` are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 temperature,
                 logits,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.BinConcrete(
            temperature,
            logits,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(BinConcrete, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)


BinGumbelSoftmax = BinConcrete


class ExpConcrete(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of ExpConcrete `StochasticTensor` from (Maddison, 2016),
    transformed from :class:`~Concrete` by taking logarithm.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    .. seealso::

        :class:`~BinConcrete` and :class:`~Concrete`

    :param temperature: A 0-D `float` Tensor. The temperature of the relaxed
        distribution. The temperature should be positive.
    :param logits: A N-D (N >= 1) `float` Tensor of shape (...,
        n_categories). Each slice `[i, j, ..., k, :]` represents the
        un-normalized log probabilities for all categories.

        .. math:: \\mathrm{logits} \\propto \\log p

    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        `StochasticTensor` are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 temperature,
                 logits,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.ExpConcrete(
            temperature,
            logits,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(ExpConcrete, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)


ExpGumbelSoftmax = ExpConcrete


class Concrete(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of Concrete (or Gumbel-Softmax) `StochasticTensor` from
    (Maddison, 2016; Jang, 2016), served as
    the continuous relaxation of the :class:`~OnehotCategorical`.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    .. seealso::

        :class:`~BinConcrete` and :class:`~ExpConcrete`

    :param temperature: A 0-D `float` Tensor. The temperature of the relaxed
        distribution. The temperature should be positive.
    :param logits: A N-D (N >= 1) `float` Tensor of shape (...,
        n_categories). Each slice `[i, j, ..., k, :]` represents the
        un-normalized log probabilities for all categories.

        .. math:: \\mathrm{logits} \\propto \\log p

    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param is_reparameterized: A Bool. If True, gradients on samples from this
        `StochasticTensor` are allowed to propagate into inputs, using the
        reparametrization trick from (Kingma, 2013).
    :param check_numerics: Bool. Whether to check numeric issues.
    """

    def __init__(self,
                 name,
                 temperature,
                 logits,
                 n_samples=None,
                 group_ndims=0,
                 is_reparameterized=True,
                 check_numerics=False,
                 **kwargs):
        dist = distributions.Concrete(
            temperature,
            logits,
            group_ndims=group_ndims,
            is_reparameterized=is_reparameterized,
            check_numerics=check_numerics,
            **kwargs
        )
        super(Concrete, self).__init__(None, name, dist, n_samples=n_samples,
                                       **kwargs)


GumbelSoftmax = Concrete


class Empirical(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of Empirical `StochasticTensor`.
    For any inference it is always required that the variables are observed.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param dtype: The value type of samples from the distribution.
    :param batch_shape: A `TensorShape` describing the `batch_shape` of the
        distribution.
    :param value_shape: A `TensorShape` describing the `value_shape` of the
        distribution.
    :param is_continuous: A bool or `None`. Whether the distribution is
        continuous or not. If `None`, will consider it continuous only if
        `dtype` is a float type.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    """

    def __init__(self,
                 name,
                 dtype,
                 batch_shape,
                 n_samples=None,
                 group_ndims=0,
                 value_shape=None,
                 is_continuous=None,
                 **kwargs):
        dist = special.Empirical(
            dtype,
            batch_shape,
            value_shape=value_shape,
            group_ndims=group_ndims,
            is_continous=is_continuous,
            **kwargs
        )
        super(Empirical, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)


class Implicit(StochasticTensor):
    """
    .. warning::

        Deprecated in 0.4, will be removed in 0.4.1.

    The class of Implicit `StochasticTensor`.
    This distribution always sample the implicit tensor provided.
    See :class:`~zhusuan.framework.base.StochasticTensor` for details.

    :param name: A string. The name of the `StochasticTensor`. Must be unique
        in the `BayesianNet` context.
    :param samples: A N-D (N >= 1) `float` Tensor.
    :param value_shape: A list or tuple describing the `value_shape` of the
        distribution. The entries of the list can either be int, Dimension or
        None.
    :param group_ndims: A 0-D `int32` Tensor representing the number of
        dimensions in `batch_shape` (counted from the end) that are grouped
        into a single event, so that their probabilities are calculated
        together. Default is 0, which means a single value is an event.
        See :class:`~zhusuan.distributions.base.Distribution` for more detailed
        explanation.
    :param n_samples: A 0-D `int32` Tensor or None. Number of samples
        generated by this `StochasticTensor`.
    """

    def __init__(self,
                 name,
                 samples,
                 value_shape=None,
                 group_ndims=0,
                 n_samples=None,
                 **kwargs):
        dist = special.Implicit(
            samples,
            value_shape=value_shape,
            group_ndims=group_ndims,
            **kwargs
        )
        super(Implicit, self).__init__(
            None, name, dist, n_samples=n_samples, **kwargs)
