#!/usr/bin/env python3

import json
import argparse
from urllib import request, error


# 提交答案服务域名或IP, 将在赛前告知
JUDGE_SERVER = "http://judge.aiops-challenge.com"  # 评测服务地址
# 团队ID, 登陆账号后, 选择对应的比赛, 在团队页面中可以获得。如未注明团队ID，结果不计入成绩
TICKET = "TeamID" // 填团队ID信息


def submit(data, judge_server=None, ticket=None):
    judge_server = judge_server or JUDGE_SERVER
    ticket = ticket or TICKET
    
    if not judge_server or not ticket:
        missing = ["judge_server" if not judge_server else "", "ticket" if not ticket else ""]
        missing = [m for m in missing if m]
        print("Required fields must be provided: %s" % ', '.join(missing))
        return None
    
    req_data = json.dumps({'data': data}).encode('utf-8')
    req = request.Request(judge_server, data=req_data, headers={'ticket': ticket, 'Content-Type': 'application/json'})

    try:
        with request.urlopen(req) as response:
            response_body = response.read().decode('utf-8')
            return json.loads(response_body)['submission_id']
    except error.HTTPError as e:
        msg = e.reason
        response_body = e.read().decode('utf-8')
        if response_body:
            try:
                msg = json.loads(response_body)['detail']
            except:
                pass
        print("[Error %s] %s" % (e.code, msg))

    except error.URLError as e:
        print(e.reason)
        return None

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Submit to judge server")
    parser.add_argument('result_path', nargs='?', default='result.jsonl', help='Path to the submission file, default is result.jsonl')
    parser.add_argument('-s', '--server', help='Judge server URL, if not specified, the global JUDGE_SERVER variable will be used')
    parser.add_argument('-k', '--ticket', help='Submission ticket, if not specified, the global TICKET variable will be used')
    
    args = parser.parse_args()

    try:
        with open(args.result_path, 'r') as file:
            data = [json.loads(line.strip()) for line in file if line.strip()]
    except Exception as e:
        print(e)
        exit(1)

    submission_id = submit(data, judge_server=args.server, ticket=args.ticket)
    if submission_id:
        print("Success! Your submission ID is %s." % submission_id)
        exit(0)
    else:
        exit(1)
